;
;  Copyright (c) 2003 VIA Networking, Inc. All rights reserved.
;
;  This software is copyrighted by and is the sole property of
;  VIA Networking, Inc. This software may only be used in accordance
;  with the corresponding license agreement. Any unauthorized use,
;  duplication, transmission, distribution, or disclosure of this
;  software is expressly forbidden.
;
;  This software is provided by VIA Networking, Inc. "as is" and any
;  express or implied warranties, including, but not limited to, the
;  implied warranties of merchantability and fitness for a particular
;  purpose are disclaimed. In no event shall VIA Networking, Inc. be
;  liable for any direct, indirect, incidental, special, exemplary, or
;  consequential damages.
;
;
;  File: intr.asm
;
;  Purpose:
;       This file contains entries of h/w interrupts which is invoked when tx
;       event or rx event occurs. The interrupt service routine is divided for
;       2 portions. One portion performs time critical processing, the other
;       performs not time critical processing.
;
;       In time critical portion, there are 5 steps must be done.
;
;           (1) Increment the interrupt nesting level.
;           (2) Switch to protected mode if the interrupt routine was called in real
;               mode.
;           (3) Disable the network access card's interrupt.
;           (4) Determine the reason that the interrupt routine was called.
;               . If the network access card received a frame, call protocol's
;                 RECEIVELOOKAHEAD indication. Caution must be done is making sure
;                 indication is ON before call protocol's RECEIVELOOKAHEAD indication
;
;               . If the network access card sucessfully completed a previously
;                 requested transmittion, release the usage tx buffer.
;
;           (5) Rearm the network access card's interrupt level.
;
;       In not time critical portion, there are 5 steps must be done.
;
;           (1) Generate an INDICATIONCOMPLETE
;           (2) Send EOI for next interrupt coming.
;           (3) decreament the interrupt nesting level
;           (4) switch back to real mode if the interrupt was called in real mode.
;
;  Author: GuardKuo
;
;  Date: Jan 15, 2003
;
;  Functions:
;       IntHandler
;       intr_Delay1Second
;       intr_MiniSec
;       intr_read_tick_counter
;       DriverISR
;       vPacketReceived
;       vPacketTransmitted
;       RcvLookahead
;       StatusIndicate
;       QueryAuto
;       SetToHalfDPX
;       SetToFullDPX
;       WriteMII
;       ReadMII
;       MIIDelay
;       SenseLinkUnLink
;       TurnOnOffMIIBit
;
;  Revision History:
;       01-15-2003 GuardKuo: add VIA Networking copyright message
;       01-23-2003 GuardKuo: modified MII read/write function, and add TurnOnOffMIIBit
;       06-10-2004 GuardKuo: remove TurnOnOffMIIBit. Add MIITurnOnBits and MIITurnOffBits
;
                page    88,122

                title   INTR - Interrupt Handlers (2.0.1 Spec)

.xlist
IFDEF OS2
include devhlp.inc
ENDIF
include includes.mac
.list
.lfcond
.sall

.386
;------ Exported Interfaces -------------------------------------------------;

                public  IntHandler
                public  StatusIndicate
                public  MIIRead
                public  MIIWrite
                public  MIIDelay
                public  SenseLinkUnLink

;------ Imported Interfaces -------------------------------------------------;

                DEXTRN  intsreal:WORD
                DEXTRN  intstotal:WORD
                DOSXTRN DDStkOff:WORD
                DOSXTRN DDStkSeg:WORD
                DOSXTRN DDTopStk:BYTE
                OS2XTRN devHelper:DWORD
                OS2XTRN segTX:WORD

                extrn   Ioport:word
                extrn   cs_ds:word
                IFNDEF DYNAALLOC
                extrn   LookAheadBuf: byte
                ENDIF
                extrn   TotalRxCnt:word
                extrn   LookAheadSize:word
                OS2XTRN specific:word
                extrn   EOIs:WORD
                extrn   ICpending:BYTE
                extrn   IMRport:WORD
                extrn   Intpending:BYTE
                extrn   common:BYTE
                extrn   devHeader:WORD
                extrn   disableMask:BYTE
                extrn   enableMask:BYTE
                extrn   indications:BYTE
                extrn   nestLevel:BYTE
                extrn   offDepth:BYTE
                extrn   pdispatch:BYTE
                extrn   protDS:WORD
                extrn   state:BYTE
                extrn   status:BYTE

                extrn   FindAddressInMCTable:near ;from util

                extrn   wBak:word

;/////////////////////////////////////////
;Added by Charles Yu 1/07/99
                extrn   PCIRevisionId:byte
;/////////////////////////////////////////

        extrn           wImrShadow:word
        extrn           wIsrImage:word
        extrn           PktRaceContinue:byte
        extrn           wTxNumPosted:word
        extrn           pdescRxRead:word
        extrn           wRxDescFilterValue:word
        extrn           pdescTxRead:word
        extrn           pdescTxWrite:word

        extrn           byMIIIndex:byte
        extrn           wMIIValue:word
        extrn           FDXFlagNow:byte
        extrn           LinkStatus:byte
        extrn           AutoFlag:byte
        extrn           FDXFlag:byte
        extrn           byUserOverride:byte

        extrn           vKickNicToWork:near
        extrn           vSetFlowCTL:near        ; added by Ben (2001/08/06)
        extrn           vUpdateTallyCounter:near

        extrn           byLineSpeed:byte        ; added by Ben (10/04/00)
        extrn           cs_ip:WORD              ; added by Ben (10/12/00)

        extrn           wStartTime:word
        extrn           wErrCurrTxDescAddr:dword
IFDEF SNAP
                DEXTRN  SnapRx:NEAR             ; from recv
                DEXTRN  SnapTx:NEAR             ; from xmit
ENDIF
        extrn           wVALPKTLENFlag:word         ; Added by guard (2003/06/21)

        IFDEF   DYNAALLOC
        extrn   RXBufferSegment:word
        extrn   CurRXDescPtr:word
        ENDIF

NEWPAGE <Code>
;------ Code ----------------------------------------------------------------;

                BEGIN_CODE


; Current packet filter mask (for ms_fltr)
;
FM_MC   equ     00001h          ;Multicast (and directed/group/functional).
FM_BC   equ     00002h          ;Broadcast.
FM_PM   equ     00004h          ;Promiscuous.
FM_SRC  equ     00008h          ;Source-routing.


;------ Switch to Protect Mode ----------------------------------------------;
;
; entered in real mode.  switch to protect mode and push an extra return
; address on the stack to switch us back when we leave.
;
IFDEF OS2
                align   4
                public  int_toprotect
int_toprotect:

                DINC    intsreal

                mov     si,offset DGROUP:devHeader
                mov     dl,DevHlp_RealToProt
                call    devHelper

                push    cs
                lea     ax, CGROUP:int_toreal
                push    ax
                jmp     short int_enter
ENDIF

NEWPAGE <IntHandler - Interrupt Handler>
;------ IntHandler ----------------------------------------------------------;
;                                                                            ;
;       This routine fields an interrupt from the adapter.                   ;
;                                                                            ;
;       For the AT bus adapters once an interrupt is generated to the PC,    ;
;       another won't be until the Interrupt Release port is hit.  This      ;
;       port is ignored by the microchannel adapters so in addition we need  ;
;       to disable interrupts from the adapter at the start of the           ;
;       interrupt handler and reenable them just before hitting the          ;
;       Interrupt Release port.  Though these steps are not necessary for    ;
;       the AT bus adapters it was easier to incorporate them directly than  ;
;       to avoid them.                                                       ;
;                                                                            ;
;       Because of this enable/disable logic, it is not necessary to do      ;
;       an implicit IndicationOff/IndicationOn around the ReceiveLookahead   ;
;       call.  The IndicationOff has already been done at the start of the   ;
;       interrupt handler.  On return if the indications flag has been       ;
;       cleared, we leave offDepth set to one.  Then when we go to rearm     ;
;       the adapter, we reenable interrupts only if offDepth is zero.        ;
;                                                                            ;
;       Possible Optimizations:                                              ;
;                                                                            ;
;       1) It is possible to field many interrupts without mode switching    ;
;       if we are in the compatibility box.  Those interrupts which result   ;
;       in calls to the protocol need to switch, but many interrupts do not  ;
;       (a transmit proceeds through several interrupts and if the request   ;
;       handle is zero, none of these result in calls to the protocol).  To  ;
;       make the driver bimodal we need to make the selectors segMMIO and    ;
;       segSM bi-modal, by setting them to the appropriate values for the    ;
;       current mode.  Most of the code is then bimodal automatically.  The  ;
;       only tricky logic is in handling the transmit ARB.                   ;
;                                                                            ;
;       While the theory is that this will increase the system performance   ;
;       and network throughput while in the real mode box, early tests do    ;
;       not show any significant change.  Most interrupts are taken in       ;
;       protect mode even when running a tight loop in the real mode box.    ;
;       So for the moment since they significantly complicate things, the    ;
;       bimodal changes have been backed off.                                ;
;                                                                            ;
;       2) At the end of the interrupt handler just before returning check   ;
;       the ISRP to see if there is something else that has come up and      ;
;       take care of it rather than returning from the interrupt handler     ;
;       and coming back in.  Some care must be taken to avoid issuing extra  ;
;       EOIs and such but with a real protocol stack that imposes some       ;
;       delay between the rearming of the adapter and the end of the         ;
;       interrupt handler it seems like between 15% and 30% of the           ;
;       interrupts (depending on processor speed) could be looped around in  ;
;       this fashion.  What exactly the impact would be is not clear.  A     ;
;       couple of quick hacks at it introduced bugs which I didn't have      ;
;       time to work around.                                                 ;
;                                                                            ;
;----------------------------------------------------------------------------;

;------ Common Interrupt Handler --------------------------------------------;

IFDEF  RxDebug
cxsave   dw  0
essave   dw  0
disave   dw  0
ENDIF

IFDEF   DOS
 IFDEF  SHOW_KEY
include    show.asm
 ENDIF
ENDIF

        ;VT3065A : Define Variables
        ;wErrCurrTxDescAddr dd  0
        ;VT3065A : Define Variables

                align   4
IntHandler      proc    far

        ; disable interrupts explicitly
        ; -------------------------------
        cli

        ; under DOS we must save the registers ourselves and set up DS to equal CS
        ; since there is only one segment (both handled by OS/2 for us)
        ; --------------------------------------------------------------
IFDEF DOS
        PUSHR   eax,ebx,ecx,edx,esi,edi,es,ds,ebp
        mov     ax, cs                  ; local data
        mov     ds, ax

        cmp     nestLevel,2                 ;1111
        jae     int_return                  ;1111

        cmp     nestLevel,0                 ;1111
        jnz     UseTheirStack               ;1111

        ; under DOS, switch stacks
        ; -------------------------------
        mov     DDStkSeg,ss                 ; save stack segment
        mov     DDStkOff,sp                 ; save stack pointer
        mov     ss,ax                       ; new stack segment
        mov     sp,offset DGROUP:DDTopStk   ; new stack pointer
UseTheirStack:
ENDIF   ;   end IFDEF DOS

        ; if in real mode, switch to protected mode (note that this test is not
        ; sufficient if "OS/3" uses the Protected Virtual 8086 mode).
        ; --------------------------------------------------------------
IFDEF OS2
        DINC    intstotal

        smsw    ax                      ; ax <- machine status word
        and     ax,1                    ; protect mode bit (faster than shr)
        jz      int_toprotect

        public  int_enter
int_enter:
        mov     ds, cs:cs_ds
ENDIF

        ; increment the interrupt nest level
        ; -------------------------------
        inc     nestLevel

        ; clear the direction flag to conform with Microsoft C calling conventions
        ; --------------------------------------------------------------
        cld

        ; disable interrupts from the adapter through the PIC IMR
        ; --------------------------------------------------------------
        inc     offDepth

        mov     dx,IMRport
        in      al,dx
        slow
        slow
        or      al,disableMask
        out     dx,al

        ; allow interrupt controller to acknowledge other system interrupts.  under
        ; DOS this is always done by playing directly with the PIC.  under OS/2 we
        ; use the DevHlp call unless FASTEOI is defined in which case we also do
        ; the EOI by playing with the PIC directly.  This is because the DevHlp call
        ; is horrendously inefficient (125 instructions or about 150usec on an 8MHz
        ; AT).  Defining FASTEOI makes a noticable performance difference.
        ;--------------------------------------------------------------
IFDEF DOS

ELSEIFDEF FASTEOI
        mov     ax,EOIs                 ; commands to issue
        out     INT_2EOIR,al            ; command to master
        xchg    ah,al
        out     INT_1EOIR,al            ; command to slave
        slow
        slow
ELSE
        mov     al,specific.mc_interrupt.lobyte
        mov     dl,DevHlp_EOI
        call    devHelper
ENDIF
        ; added by Ben (10/12/00), we need to test the
        ; interrup was invoked by which NICs.
        ; ---------------------------------------------S
        call    DriverISR
        cmp     ax, 0
        je      Old_NIC
        jmp     IsOurInterrupt
Old_NIC:
        ; fixed by guard(2004/08/31)
        ; simulate int n
        ; becasue call function only push cs and ip to stack,
        ; so we need push flag by ourself.
        sti
        pushf
        call    DWORD PTR cs_ip         ; Call DriverISR
IFDEF OS2
        popf
ENDIF
IsOurInterrupt:
        ; ---------------------------------------------E

        cli
        mov     indications, 0ffh
        public  int_rearm
int_rearm:
        dec     offDepth

        mov     dx,IMRport
        in      al,dx

        slow
        slow

        and     al,enableMask
        out     dx,al

ir_exit:
IFDEF DOS
        cmp     nestLevel,1                 ;1111
        jnz     UseTheirStack1              ;1111
        mov     ss,DDStkSeg                 ; restore old SS
        mov     sp,DDStkOff                 ; restore old SP
UseTheirStack1:
ENDIF

        cmp     nestLevel,1
        ja      int_exit
int_postloop:
        cmp     ICpending,YES
        je      int_postic

        public  int_exit
int_exit:
        dec     nestLevel


        ; switch back to the original stack. There is no stack supplied to the Protocol
        ; for indication completes, and confirmations.
        ;--------------------------------------------------------------


        ; return from interrupt.
        ;-------------------------------
        public  int_return
int_return:
IFDEF OS2

        ; if we entered in real mode this will just return to int_toreal because of
        ; the extra return address on the stack.
        ;--------------------------------------------------------------
        clc
        ret

        ; return to real mode, and really return from interrupt
        ;--------------------------------------------------------------
        public  int_toreal
int_toreal:
        mov     si,offset DGROUP:devHeader
        mov     dl,DevHlp_ProtToReal
        call    devHelper
        clc
        ret
ELSE

        ; pop off registers that were saved at the interrupt routine entry on the
        ; existing stack
        ;--------------------------------------------------------------
        mov     ax,EOIs                 ; commands to issue
        out     INT_2EOIR,al            ; command to master
        slow
        slow
        xchg    ah,al
        out     INT_1EOIR,al            ; command to slave
        slow
        slow
        POPR    eax,ebx,ecx,edx,esi,edi,es,ds,ebp
        iret
ENDIF

        ;------ Indication Complete -----------------------------------
        ;
        ; turn interrupts on and generate an IndicationComplete to the protocol.
        ; then around the post-processing loop again.
        ;
        ; interrupts are disabled.
        ;--------------------------------------------------------------
        align   4
        public  int_postic

int_postic:
IFDEF   DOS
        IFDEF  SHOW_KEY
        ENDIF
ENDIF
        mov     ICpending,NO

        sti
        push    common.cc_ID
        push    protDS
        call    pdispatch.pd_indcomplete
        cli

        mov     ds, cs:cs_ds
        jmp     int_postloop

IntHandler      endp


;--------------------------------------------------------------
; This routine delays for 1 second.
;--------------------------------------------------------------
        public      intr_Delay1Second
intr_Delay1Second    proc    near
        push    ecx
        mov     ecx, 100
intr_wait1second:
        call    intr_MiniSec
        loop    intr_wait1second
        pop     ecx
        ret
intr_Delay1Second    endp

;--------------------------------------------------------------
; This routine delays for 10 mini seconds.
;--------------------------------------------------------------
intr_MiniSec proc    near
        push    ax
        call    intr_read_tick_counter
        mov     wStartTime, ax
intr_wait1:
        call    intr_read_tick_counter
        neg     ax
        add     ax, wStartTime
        cmp     ax, 23866
        jbe     intr_wait1
        pop     ax
        ret
intr_MiniSec endp

;--------------------------------------------------------------
; This routine read 8254 time tick
;--------------------------------------------------------------
intr_read_tick_counter proc  near
        xor     ax, ax                  ; Clear accumulator.
        mov     al, 06h
        out     43h, al                 ; Command 8254 to latch T0's count
        in      al, 40h                 ; Read LSB first
        mov     ah, al
        in      al, 40h                 ; Read MSB
        xchg    al, ah                  ; Get in proper order
        ret
intr_read_tick_counter endp

; -----------------------------------------------------
;
;
; -----------------------------------------------------
DriverISR       proc    near

        push    ebp

        mov     bx, Ioport

        lea     dx, [bx].SCsrRegStruc.IMR0
        xor     ax,ax
        out     dx,ax

PollAgain:
        lea     dx, [bx].SCsrRegStruc.ISR0
        in      ax, dx                      ; AX = ISR status
        test    ax, wImrShadow              ; Check if we want this
        out     dx, ax                      ; Clear all ISR active bits
        out 97h, al
        ; added by Ben (10/12/00)
        ; --------------------------------------------S
        jz      NotOurISR
        ; ---------------------------
        ; added by guard(10/09/2003)
        ; ----------------------------------
        cmp     ax, 0FFFFh
        je      NotOurISR

        jmp     OurISR

NotOurISR:
        lea     dx, [bx].SCsrRegStruc.IMR0
        mov     ax, wImrShadow
        out     dx, ax
        pop     ebp
        mov     ax, 0                       ; Notify not our interrupt
        ret
OurISR:
        ; --------------------------------------------E

        mov     wIsrImage, ax

        ;---------------------------------------------------------
        ; Check if we received a packet through the RX descriptor
        ;---------------------------------------------------------
        mov     bx, pdescRxRead
        mov     bp, [bx].SRxDescStruc.wRxLength

IFDEF   DOS
        IFDEF   SHOW_KEY
        show     bp, 5
        ENDIF
ENDIF

        test    bp, W_RX_LENGTH_OWN         ; check if host side owns this
                                            ; descriptor(bit cleared).
        jne     short ProcTxEvent
        call    vPacketReceived

ProcTxEvent:
        ;------------------------------------------------------------
        ; Check if we transmitted a packet through the TX descriptor
        ;------------------------------------------------------------
        cmp     wTxNumPosted, 0
        je      short ProcAbnormalCond

        mov     bx, pdescTxRead
        test    [bx].STxDescStruc.wTxOwn, W_TX_OWN_BIT
                                            ; check descriptor's ownnership
        jnz     short ProcAbnormalCond   ; to be host owned (bit cleared).

        call    vPacketTransmitted

ProcAbnormalCond:
        ;-----------------------------------------------------
        ; Check if there is an abnormal condition interrupts?
        ;-----------------------------------------------------
        test    wIsrImage, W_ISR_ABNORMAL
        jnz     short AbnormalInterrupt

        cli

ISR_exit:
        mov     bx, Ioport
        lea     dx, [bx].SCsrRegStruc.IMR0
        mov     ax, wImrShadow
        out     dx, ax

        ; added by Ben (10/12/00)
        ; --------------------------------------------------------------
        mov     ax, 1                       ; Notify it's our interrupt
        pop     ebp
        ret

AbnormalInterrupt:

        mov     bx, Ioport

        ; No Rx Buff error? A single-buffer packet can NOT capture desc.
        ;---------------------------------------------------------------
        test    wIsrImage, W_ISR_SRC

        ;//////////////////////////////
        ;Modified by Charles Yu 1/20/99
        jz      ANI000
        ;//////////////////////////////

        ; 1. Check if Link OK
        ; 2. If Link OK, check Full or Half Duplex.
        ; 3. If Link Fail, issue re-AutoNegotyiation, then leave.
        ;        int     3;int3
        ;        nop
        ;        xor     LinkStatus, 1             ; Toggle...
        ; ------------------------------------------------------
        call    SenseLinkUnLink
        cmp     LinkStatus, LinkOK          ; Toggle...
        jnz     NotLinkNow

        ; modified by Ben (2001/07/24), [bx].byUserOverride->byUserOverride
        ; ------------------------------------------------------------------
        cmp     byUserOverride, 1           ; user specify SPEED or Duplex keywork
        jz      IsrSrcOver

        call    QueryAuto

        ;-----------------------------
        ; Added by Charles Yu 3/01/99
        ; Delay DAVICOM LEGACY PHY
        ;-----------------------------
        call    intr_Delay1Second
        cmp     FDXFlagNow, 0
        je      SetHalf2
        call    SetToFullDPX
        jmp     FDXOver2

SetHalf2:
        call    SetToHalfDPX

FDXOver2:
        ; add by Ben (2001/08/10), for settiing flow control registers
        ; -------------------------------------------------------------
        call    vSetFlowCTL

        ; very bad patch way !!!!!
        ; Compiler will make .386 entry dword
        ; ----------------------------------------
IFDEF   OS2
        add     esp, 2
ENDIF
        jmp     IsrSrcOver

NotLinkNow:

        ;-------------------------------------
        ;Added by Charles Yu 1/08/99
        ;if Auto-Nego
        ;  if 3043
        ;    reset PHY
        ;----------------------------------
        cmp     AutoFlag, 1
        jnz     IsrSrcOver

        ;Is Auto_Nego
        ; ---------------
        cmp     pciRevisionId, 20h
        jge     IsrSrcOver

        ;Is 3043
        ;Reset chip
        ; ------------------------
        mov     byMIIIndex, MII_REG_BMCR
        mov     wMIIValue, BMCR_RESET
        call    MIITurnOnBits

IsrSrcOver:

ANI000:
        ; No Rx Buff error? A single-buffer packet can NOT capture desc.
        ;---------------------------------------------------------------
        test    wIsrImage, W_ISR_NRB
        jz      short ANI010

ANI010:
        ; Rx buffer unavailable?
        ; A multiple-buffers packet can NOT capture next desc.
        ;----------------------------------------------------------
        test    wIsrImage, W_ISR_RU
        jz      short ANI020


ANI020:
        ; Tx buffer unavailable? The same as TBUFF in TD.
        ;-------------------------------------------------
        test    wIsrImage, W_ISR_TU
        jz      short ANI030

        push    bx                          ; synchrozsize HW and SW pointer
        lea     dx, [bx].SCsrRegStruc.CurrTxDescAddr
        in      eax, dx
        xor     edx, edx
        mov     dx, ds
        shl     edx, 4
        sub     eax, edx
        mov     bx, ax
        mov     pdescTxWrite, bx            ; BX Point at Tx curr desc.
        mov     pdescTxRead, bx             ; BX Point at Tx curr desc.
        pop     bx

ANI030:
        ; Tally counter overflow?
        ;-----------------------------
        test    wIsrImage, W_ISR_CNT
        jz      short ANI030_1

        call    vUpdateTallyCounter         ; read Tally counter to reset it.

        ; very bad patch way !!!!!
        ; Compiler will make .386 entry dword
        ; ----------------------------------------
IFDEF   OS2
        add     esp, 2
ENDIF

ANI030_1:
        ; Transmit error summary?
        ;-----------------------------
        test    wIsrImage, W_ISR_TXE
        jz      short ANI040

        lea     dx, [bx].SCsrRegStruc.CommandReg
        in      ax, dx
        or      ax, W_CR_TXON OR W_CR_RXON OR W_CR_START
        out     dx, ax

ANI040:
        ; FIFO packet race?
        ;-----------------------------
        test    wIsrImage, W_ISR_PRA
        jz      short ANI050
                                            ; specific counter for it.
        inc     PktRaceContinue

ANI050:
        ; System bus error?
        ;--------------------
        test    wIsrImage, W_ISR_BE
        jz      short ANI060

        cli
ANI060:
        call    vKickNicToWork              ; NIC is in stop mode.

        ; very bad patch way !!!!!
        ; Compiler will make .386 entry dword
        ;--------------------------------------------------------------
IFDEF   OS2
        add     esp, 2
ENDIF
                                            ; Kick it to work.
        jmp     Isr_Exit
DriverISR       endp

; -----------------------------------------------------------
;
;
;
; -----------------------------------------------------------
        align   4
vPacketReceived  proc    near
        push    es
        push    ebp

HandleAnotherReceive:

        push    bx                                  ; save the pointer to the current RX
                                                    ; descriptor on the stack
        mov     ds, cs:cs_ds
        IFDEF   DYNAALLOC
        mov     ax, [bx].SRxDescStruc.pRxLogicSeg
        mov     es, ax
        mov     di, [bx].SRxDescStruc.pRxBuffer
        ELSE
        push    ds
        pop     es                                  ; ES=DS
        lea     di, [bx].SRxDescStruc.abyRxBuffer   ; ES:SI -> lookahead
        ENDIF

        test    status.ms_filter, FM_PROMISCUOUS
        jnz     PacketFilterDone                    ; receive all packet
        mov     ax, word ptr es:[di]
        and     al, ah
        cmp     al, 0FFh                            ; is broadcast packet?
        jne     IsMCPkt
        test    status.ms_filter, FM_BC
        jnz     PacketFilterDone

IsMCPkt:
        test    status.ms_filter, FM_MC
        jz      InvalidMCTable

        test    byte ptr es:[di], 1                 ; is multicast packet?
        jnz     @F
        jmp     short PacketFilterDone

InvalidMCTable:
        jmp     FinishUpReceive

@@:
        push    bx
        push    di
        mov     ax, word ptr es:[di]
        mov     bx, word ptr es:[di+2]
        mov     di, word ptr es:[di+4]

        call    FindAddressInMCTable

        ; very bad patch way !!!!!
        ; Compiler will make .386 entry dword
        ;--------------------------------------------------------------
IFDEF   OS2
        add     esp, 2
ENDIF

        pop     di
        pop     bx
        or      dl, dl                      ; set zero flag, which is
                                            ; cleared by patch code

        SJNZ    DiscardThisPacket

PacketFilterDone:

        IFDEF   DYNAALLOC
        push    ds
        pop     es
        ENDIF
        ; marked by Ben (2001/08/31), coz under VT3106, if rx packet is
        ; tagged, then MAC will extract the tag field (4 bytes) and append to
        ; the RD, and the RD's packet length will be decreased 4!!! So if it's
        ; a ICMP (PING) packet, then the original packet length will be 60 bytes
        ; (include TAG field), and the BP will be 56 after MAC extracted this packet.
        ; ---------------------------------------------------------------------------
        sub     bp,4                        ; Subtract the rx packet CRC
        mov     TotalRxCnt, bp

        ; marked by Ben (01/19/2001), for conforming 802.3 spec
        ; --------------------------------------------------------S
        cmp     bp, 1514
        ja      DiscardThisPacket
        ; --------------------------------------------------------E

        ; add by Ben (2001/08/31), it's only for VT3065 or VT3043
        ; to check the packet length if < 60 here!!!
        ; --------------------------------------------------------S
        cmp     pciRevisionId, RevisionID3106J
        jae     SkipTestLen
        ; --------------------------------------------------------E

        cmp     bp, 60
        jb      DiscardThisPacket

SkipTestLen:                                ; add by Ben (2001/080/31)
        cmp     bp, LookAheadSize
        jbe     @F
        mov     bp, LookAheadSize

@@:
        mov     ax, [bx].SRxDescStruc.wRxStatus

        ; See if descriptor is the last of a packet!
        ; Only the last one has valid status.
        ;-------------------------------------------
        test    ax, W_RX_STATUS_EDP
        je      RxNotLast

        ; The last descriptor:
        ; It's the last descriptor of a packet because only the last
        ; has valid status. Let's check errors if any.
        ;----------------------------------------------------------
        test    ax, W_RX_STATUS_ABNORMAL
        jz      NoReceiveErrors

        ;--------------------------
        ; Handle receive errors
        ;--------------------------
ReceiveErrors:
        ; Errors generated by NIC:
        ; We handle BUFF|SERR error in DriverIsr
        ; ----------------------------------------
        test    ax, W_RX_STATUS_FOV         ; FIFO overflow
        jz      short RXE010

RXE010:
        ; Abnormal status NOT generated by our NIC:
        ; Check if runt frame (shorter than 64 bytes)
        ;-------------------------------------------
        test    ax, W_RX_STATUS_RUNT
        jz      short RXE020

RXE020:
        ; Check if too long packet (over 1518 bytes)
        ; marked by Ben (2003/03/01), W_RX_STATUS_LONG
        ; had been removed from abnormal checking. Coz
        ; we support tag packet (the max = 1522)
        ;---------------------------------------------
        ;test    ax, W_RX_STATUS_LONG
        ;jz      short RXE030

RXE030:
        test    ax, W_RX_STATUS_CRC
        jz      short RXE040

RXE040:
        test    ax, W_RX_STATUS_FAE
        jz      RXE080

RXE080:
        ;!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!  12345  !!!!!!!
        jmp     ReturnCurrentBuffer         ;!!!!! Left to be determined
        ;!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!  12345  !!!!!!!

        ; Add by Ben (2001/08/06):
        ;   Check RXLERR bit in RD for VT3106
        ; --------------------------------------------\
        ; add by Ben (2001/08/06), VT3106 to check length error
        ; -----------------------------------------------------
NoReceiveErrors:
        cmp     pciRevisionId, RevisionID3106J
        jb      VT3065_CHKLEN

        ; ------------------------------
        ;       VT3106
        ; ------------------------------
        PUSHR   eax,ebx,ecx,edx,esi,edi,es,ds,ebp

        ; -------------------------------------------------------------------
        ; Check the packet length on RD descriptor,
        ; if ECX <= 60 when it's a tagged packet, then adjusted to 64
        ; (include CRC)!!!. Otherwise, Novell OS will discard this packet.
        ; -------------------------------------------------------------------
        mov     ax, TotalRxCnt
        cmp     ax, 60
        jbe     IfAdjustPacketLength
        jmp     IfLayer2CtlPkt

        ; RD's packet (TAGGED) length <= 60!!!
        ; ------------------------------------
IfAdjustPacketLength:
        test    [bx].SRxDescStruc.bPQSTS, B_RX_PQSTS_TAG
        jz      IfLayer2CtlPkt

        ; It's TAG RD packet
        ; -------------------
        mov     ax, 60                  ; not include CRC
        mov     TotalRxCnt, ax          ; Update variable

        ; -------------------------------------------
        ; Drop the untagged packet if VID is setting
        ;   (exclude layer 2 control packets)
        ;   (It won't be happened in NDIS2 ODI)
        ; -------------------------------------------
IfLayer2CtlPkt:
        ; -------------------------------------------
        ; Drop the packet if RXLER or SNAP packet
        ; -------------------------------------------
        test    [bx].SRxDescStruc.bPQSTS, B_RX_PQSTS_SNAPTAG
        jnz     SNAPTAGPKT

        ; marked by Ben (2003/03/01), if this happened,
        ; we still take this packet to upper layer (this
        ; is option in driver spec)
        ; ---------------------------------------------------
        test    [bx].SRxDescStruc.bPQSTS, B_RX_PQSTS_RXLER
        jnz     RxLENErrPKT

NormalPKT:
        jmp     NoDropThisPKT

RxLENErrPKT:
        jmp     CheckVALPKTLEN
        ; It's SNAP+TAG
        ; ----------------------------
SNAPTAGPKT:
        mov     ax, TotalRxCnt          ; AX = Incoming Frame Size
        sub     ax, 14                  ; AX = Actual Frame Size (CRC had been cut)
                                        ;   (without :
                                        ;       DA = SA = 6 bytes
                                        ;       CRC = 4 bytes,
                                        ;       TYPE = 2 bytes)
IFDEF   DYNAALLOC
        push    es
        mov     dx, RxBufferSegment
        mov     es, dx
        mov     si, [bx].SRxDescStruc.pRxBuffer
ELSE
        lea     si, [bx].SRxDescStruc.abyRxBuffer
ENDIF
        mov     ch, es:[si + 12]        ; get L/T field of incoming pkt
        mov     cl, es:[si + 13]
IFDEF   DYNAALLOC
        pop     es
ENDIF
        cmp     cx, ax                  ; CX=L/T, AX=actual frame size
        je      NoDropThisPKT           ; frame length field (L/T)
                                        ; <> actual frame length. Drop.
VT3065_CHKLEN:
        ; --------------------------------------------/

        ; ------------------------------
        ;       VT3065 & VT3043
        ; ------------------------------
        ; added by Ben (01/19/2001), check L/T field for conforming
        ; 802.3 spec. (The above is marked. No used)
        ; ------------------------------------------------------------------S
        PUSHR   eax,ebx,ecx,edx,esi,edi,es,ds,ebp

        ; marked by Ben (02/01/2001), BP could be LookAhead size (60)
        ; -----------------------------------------------------------------
        ;mov     ax, bp                  ; AX = Incoming Frame Size

        ; added by Ben (02/01/2001), we should use TotalRxCnt to get original
        ; packet length.
        ; ------------------------------------------------------------------
        mov     ax, TotalRxCnt          ; AX = Incoming Frame Size
        sub     ax, 14                  ; AX = Actual Frame Size (CRC had been cut)
                                        ;   (without :
                                        ;       DA = SA = 6 bytes
                                        ;       CRC = 4 bytes,
                                        ;       TYPE = 2 bytes)
        ; ------------------------------------
        ;       Get L/T field
        ; ------------------------------------
IFDEF   DYNAALLOC
        push    es
        mov     dx, RxBufferSegment
        mov     es, dx
        mov     si, [bx].SRxDescStruc.pRxBuffer
ELSE
        lea     si, [bx].SRxDescStruc.abyRxBuffer
ENDIF
        mov     ch, es:[si + 12]        ; get L/T field of incoming pkt
        mov     cl, es:[si + 13]
        mov     dh, es:[si + 14]        ; get next word of L/T field
        mov     dl, es:[si + 15]
IFDEF   DYNAALLOC
        pop     es
ENDIF
        ; ------------------------------------
        ;  Check if L/T field is LEN or TYPE
        ; ------------------------------------
        ; if frame length is >= 46 and <= 1500
        ; -----------------------------------
        cmp     cx, 46                  ; CX=L/T field
        jb      NoDropThisPKT           ; <46, no drop

        cmp     cx, 1500                ; CX = L/T field
        ja      NoDropThisPKT           ; >1500, no drop

        ; ----------------------------------------------------------
        ;               L/T field = Len
        ; ----------------------------------------------------------
        ; and this is IEEE 802.3/IEEE 802.3 SNAP frame (exclude
        ; Novell's Ethernet 802.3 frame), but frame length field is
        ; inconsistent with actual frame length, then drop it!
        ; ----------------------------------------------------------
        cmp     dx, 0FFFFh              ; DX=next word of L/T field
        je      NoDropThisPKT           ; if exclude Novell's Ethernet
                                        ; 802.3 frame, continue check.


        cmp     cx, ax                  ; CX=L/T, AX=actual frame size
        je      NoDropThisPKT           ; Not Novell's header, but
                                        ; frame length field (L/T)
                                        ; <> actual frame length. Drop.


CheckVALPKTLEN:
        cmp     wVALPKTLENFlag, 0       ; check the status of the option VALPKTLEN,
        je      NoDropThisPKT           ; if this option is off, don't check packet length
                                        ; the default is off

        ; Drop this packet
        ; --------------------------
DropThisPKT:


        POPR    eax,ebx,ecx,edx,esi,edi,es,ds,ebp
        jmp     DiscardThisPacket               ; TBD !!!

NoDropThisPKT:
        POPR    eax,ebx,ecx,edx,esi,edi,es,ds,ebp

        ; ------------------------------------------------------------------E

        mov     PktRaceContinue, 0

        ; It is a normal packet falling in a single desc. buffer.
        ;-----------------------------------------------------------
        test    ax, wRxDescFilterValue  ; Screen out error condition:
        jnz     DiscardThisPacket       ;   MSM can handle CRC, Alignment, Runt,
                                        ;   and Long errors, so we have to pass
                                        ;   these packets to MSM; however, other
                                        ;   errors generated by our NIC can NOT
                                        ;   be handed to upper layer even in
                                        ;   Promiscuous mode.

        test    ax, W_RX_STATUS_STP     ; Is it a too big packet?
        je      RxNotFirstButLast

RxFirstAndLast:
        ;------------------------------------------------------------
        ; DS - CGroup
        ; BX - current RX descriptor (pdescRxRead) &  owned by host
        ; BP - frame length field in RX descriptor
        ;------------------------------------------------------------

        mov     cx, TotalRxCnt
        add     status.ms_total_recv.loword,1
        adc     status.ms_total_recv.hiword,0

        add     status.ms_total_rbytes.loword,cx
        adc     status.ms_total_rbytes.hiword,0

        IFDEF   DYNAALLOC
        push    es
        mov     CurRXDescPtr, bx
        mov     di, [bx].SRxDescStruc.pRxBuffer
        mov     ax, [bx].SRxDescStruc.pRxLogicSeg
        mov     es, ax
        ELSE
        lea     di, DGROUP:LookAheadBuf             ; 2048 bytes long
        push    es
        push    di
        push    cx
        lea     si, [bx].SRxDescStruc.abyRxBuffer   ; ds:si -> lookahead
        and     ecx, 0ffffh
        add     cx, 3
        shr     cx, 2                               ; divide by 4
        rep     movsd                        ; [ds:si]->[es:di], dword size
        pop     cx
        pop     di
        pop     es
        ENDIF


        mov     ax, cx                              ; the lookahead buffer counts

        call    RcvLookahead
        IFDEF   DYNAALLOC
        pop     es
        ENDIF
FinishUpReceive:
        jmp     ReturnCurrentBuffer

RxNotFirstButLast:
RxNotLast:
DiscardThisPacket:

ReturnReceivedBuffer:

ReturnCurrentBuffer:
        pop     bx                          ; BX -> current RX descriptor


        ; Return the buffer to the NIC:
        ;   set descriptor's ownership bit, and clear length field
        ;---------------------------------------------------------------
        mov     [bx].SRxDescStruc.wRxLength, W_RX_LENGTH_OWN

        ; add by Ben (2001/08/06), update RXRDU of FlowCR0 by 1
        ; -----------------------------------------------------------\
        cmp     pciRevisionId, RevisionID3106J
        jb      SetRXRDUDone

        ; Add 1 to FlowCR0
        ; -----------------
        push    ax
        push    bx
        mov     bx, Ioport
        lea     dx, [bx].SCsrRegStruc.FLOWCR0
        mov     al, 1
        out     dx, al                      ; add 1 for FlowCR0
        pop     bx
        pop     ax
SetRXRDUDone:
        ; -----------------------------------------------------------/

        ; Point at next RX descriptor to be processed:
        ;---------------------------------------------------------------
        mov     bx, [bx].SRxDescStruc.pdescRxNext

        ; Check if received another packet
        ;-----------------------------------
        mov     bp, [bx].SRxDescStruc.wRxLength
        test    bp, W_RX_LENGTH_OWN     ; check descriptor's ownnership:
                                        ; If host owned (bit cleared), handle
        jz      HandleAnotherReceive    ; it again.

NoMoreReceives:
        ; Save The descriptor to be processed when called again and exit
        ;----------------------------------------------------------------
        mov     pdescRxRead, bx
        pop     ebp
        pop     es
        ret

vPacketReceived  endp

;****************************************************************************
; Packet Transmitted begin
;****************************************************************************
vPacketTransmitted       proc    near

        cli

        mov     si, Ioport

        ;VT3065A : vPacketTransmitted
        ;-------------------------------
        cmp     pciRevisionId, RevisionID3065
        jb      Chip3043vPacketTransmitted

        ; ----------------------------------
        lea     dx, [si].SCsrRegStruc.MISR
        in      al, dx
        test    al, 08h
        jz      Chip3065IntrNoWBR

        lea     dx, [si].SCsrRegStruc.CommandReg
Chip3065TXEWBRK:
        in      ax, dx
        test    ax, W_CR_TXON
        jnz     Chip3065TXEWBRK

        ;after TXON off, clear MISR_WBR
        ;-------------------------------
        lea     dx, [si].SCsrRegStruc.MISR
        mov     al, 08h                     ;clear MISR_WBR
        out     dx, al

        lea     dx, [si].SCsrRegStruc.ISR0
        mov     ax, W_ISR_INIT            ;clear W_ISR_KEYI
        out     dx, ax

ProcessCurrTxDescAddr:

      ; Read the Current TD
      ;--------------------
        lea     dx, [si].SCsrRegStruc.CurrTxDescAddr
        in      eax, dx
        mov     wErrCurrTxDescAddr, eax

WriteBackRaceLoop:

        mov     eax, [bx].STxDescStruc.pdescTxCurrPhy
        cmp     eax, wErrCurrTxDescAddr

        je      ExitWriteBackRaceLoop

        dec     wTxNumPosted                ; Mark that a packet has been
                                            ; Transmitted.

        ; Clear ownership of TD
        ;----------------------
        mov     [bx].STxDescStruc.wTxOwn, 0000h     ; Set RD to be Host-owned
        mov     [bx].STxDescStruc.wTxStatus, 0000h

      ; Point at next Tx dirty descriptor
      ;-----------------------------------
ByteAlignDesc3065Chip_0:
        test    [bx].STxDescStruc.wTCR, W_TX_TCR_EDP
        jnz     short ExitByteAlignDesc3065Chip_0
        mov     bx, [bx].STxDescStruc.pdescTxNext
        jmp     short ByteAlignDesc3065Chip_0
ExitByteAlignDesc3065Chip_0:

        mov     bx, [bx].STxDescStruc.pdescTxNext

        ;VT3065A : Return TCB
        jmp     WriteBackRaceLoop

ExitWriteBackRaceLoop:
        mov     eax, [bx].STxDescStruc.pdescTxCurrPhy
        lea     dx, [si].SCsrRegStruc.CurrTxDescAddr
        out     dx, eax

        lea     dx, [si].SCsrRegStruc.CommandReg
        in      al, dx
        or      al, W_CR_TXON
        out     dx, al
        inc     dx
        in      al, dx
        or      al, W_CR_TDMD
        out     dx, al
        jmp     CheckAnotherTransmit

Chip3065IntrNoWBR:

Chip3043vPacketTransmitted:

MoreTransmits:
        ; Update Tx counters
        ;--------------------
        cli

        ; check here for TX errors
        ;--------------------------
        mov     ax, [bx].STxDescStruc.wTxStatus

        test    ax, W_TX_STATUS_ABNORMAL
        jnz     TransmitErrors
        jmp     NoTransmitErrors

        ;-------------------
        ; Tx Error Handling
        ;-------------------
TransmitErrors:
        ; is it an error that is reported in Error summary?
        ;---------------------------------------------------
        test    ax, W_TX_STATUS_TERR
        jz      DoneWithErrorSummaryErrors

        ; Note that if errors handled in DriverIsr then the errors should NOT
        ; be handled here.
        ; In case of TX undeflow, what state does TX go?
        ;--------------------------------------------------
        test    ax, W_TX_STATUS_UDF
        jz      short TXE010

      ;AdjustTxThresh
      ;--------------
      push    ax
      lea     dx, [si].SCsrRegStruc.RXCR
      in      ax, dx
      test    ax, 8000h       ; already store and forward
      jnz     NoAdjustNow
      add     ah, 20h         ; add up one class
      out     dx, ax
NoAdjustNow:
      pop     ax

        ;---------------------------------------------------S
        ;   Added by Ben (07/10/2000)
        ;   ------------------------
        cmp     pciRevisionId, 40h
        jb      ProcessCurrTxDescAddr
        ;---------------------------------------------------E

        lea     dx, [si].SCsrRegStruc.CommandReg
Chip3065TXE010K:
        in      ax, dx
        test    ax, W_CR_TXON
        jnz     Chip3065TXE010K

        ;Handle Re-try
        ;-------------------------------

        ; Set ownership of TD
        ;----------------------
        mov     [bx].STxDescStruc.wTxOwn, W_TX_OWN_BIT

        mov     eax, [bx].STxDescStruc.pdescTxCurrPhy
        lea     dx, [si].SCsrRegStruc.CurrTxDescAddr
        out     dx, eax

        lea     dx, [si].SCsrRegStruc.CommandReg
        in      al, dx
        or      al, W_CR_TXON
        out     dx, al
        inc     dx
        in      al, dx
        or      al, W_CR_TDMD
        out     dx, al

        jmp     CheckAnotherTransmit

        ;Handle Re-try
        ;-------------------------------

        ;VT3065A : handle UDF
        ;-------------------------------
TXE010:
        ; We handle TBUFF|SERR errors in DriverIsr already.
        ; Any excessive collisions?
        ;-------------------------------------
        test    ax, W_TX_STATUS_ABT
        jz      short TXE020

        and     ax, not W_TX_STATUS_COLLISION_COUNT_MASK     ; No meaning to

        ;VT3065A : handle ABT
        ;-------------------------------
        ;---------------------------------------------------S
        ;   Added by Ben (07/10/2000)
        ;   ------------------------
        cmp     pciRevisionId, 40h
        jb      ProcessCurrTxDescAddr
        ;---------------------------------------------------E

        lea     dx, [si].SCsrRegStruc.CommandReg
Chip3065TXE020K:
        in      ax, dx
        test    ax, W_CR_TXON
        jnz     Chip3065TXE020K

        ;Handle Re-try
        ;-------------------------------
        ; Set ownership of TD
        ;----------------------
        mov     [bx].STxDescStruc.wTxOwn, W_TX_OWN_BIT

        mov     eax, [bx].STxDescStruc.pdescTxCurrPhy
        lea     dx, [si].SCsrRegStruc.CurrTxDescAddr
        out     dx, eax

        lea     dx, [si].SCsrRegStruc.CommandReg
        in      al, dx
        or      al, W_CR_TXON
        out     dx, al
        inc     dx
        in      al, dx
        or      al, W_CR_TDMD
        out     dx, al

        jmp     CheckAnotherTransmit

        ;Handle Re-try
        ;-------------------------------

TXE020:
        ; Tx buffer unavailable? The same as TBUFF in TD.
        ;-------------------------------------------------
        test    wIsrImage, W_ISR_TU
        jz      short TXE030


        ;VT3065A : handle TU
        ;-------------------------------
        lea     dx, [si].SCsrRegStruc.CommandReg
Chip3065TXE030K:

        in      ax, dx
        test    ax, W_CR_TXON
        jnz     Chip3065TXE030K

        jmp   ProcessCurrTxDescAddr

        ;VT3065A : handle TU
        ;-------------------------------
TXE030:
        ;??????????????????????????????????????????????
        ;   ReTransmit Handler ???
        ;??????????????????????????????????????????????


DoneWithErrorSummaryErrors:
        ; Transmit errors summary bit is not set in case of late colliosion,
        ; heart beat fail, in case of one or more collisions, jabber time out,
        ; and xmit abort, Now we will check these non error conditions.
        ;---------------------------------------------------------------------
        ; Any Late collision?
        ;-----------------------------------
        test    ax, W_TX_STATUS_OWC
        jz      short TXE040

TXE040:
        ; Loss of carrier?
        ;-------------------------------
        test    ax, W_TX_STATUS_CRS
        jz      short TXE050

TXE050:
        ; HEARTBEAT is meaningful only when running in AUI mode but since we
        ; use "manual" sia programing we enable hbt testing when in AUI only
        ; (so no special check is needed).
        test    ax, W_TX_STATUS_CDH
        jz      short TXE060

TXE060:

        ;VT3065A : vKickNicToWork
        ;-------------------------------
        mov     [bx].STxDescStruc.wTxStatus, 0


NoTransmitErrors:

        ;VT3065A : Add wTxNumPosted
        ;-------------------------------
        dec     wTxNumPosted            ; Mark that a packet has been transmitted;VT3065A : wTxNumPosted
        ;VT3065A : Add wTxNumPosted


        ; Check Collision count (only if there is no excessive Collisions report).
        ; We will count separately single Collision, and multiple Collisions.
        ;----------------------------------------------------------------------
        test    ax, W_TX_STATUS_COLS
        jz      DoneWithCollisionCount  ;No collision happened

        and     ax, W_TX_STATUS_COLLISION_COUNT_MASK
        jz      short DoneWithCollisionCount

        shr     ax, 1                       ; if result is 0->Collision count was 1
        jnz     short TxMultipleleCollisions

TxSingleCollision:

        jmp     short DoneWithCollisionCount
TxMultipleleCollisions:


DoneWithCollisionCount:
        ;VT3065A : Return TCB
        ;-------------------------------

      ; Point at next Tx dirty descriptor
      ;-----------------------------------
MultiDesc3065Chip:
        test    [bx].STxDescStruc.wTCR, W_TX_TCR_EDP
        jnz     ExitMultiDesc3065Chip
        mov     bx, [bx].STxDescStruc.pdescTxNext
        jmp     MultiDesc3065Chip
ExitMultiDesc3065Chip:
        mov     bx, [bx].STxDescStruc.pdescTxNext

CheckAnotherTransmit:

        ; Check For Another Transmit
        ;-----------------------------
        cmp     bx, pdescTxWrite                ; Check if there are pending
        je      short NoMoreTransmited          ; packets xmited?

        ;VT3065A: cmp wTxNumPosted, 0
        ;-------------------------------
        cmp     wTxNumPosted, 0
        je      short NoMoreTransmited          ; packets xmited?
        ;VT3065A: cmp wTxNumPosted, 0

        ; There are some pending packets transmitted.
        ;--------------------------------------------------------------
        test    [bx].STxDescStruc.wTxOwn, W_TX_OWN_BIT
                                                ; There ARE pending packets
                                                ; xmited. For now, check
                                                ; descriptor's ownnership.
        jz      MoreTransmits                   ; to be host owned (bit cleared).

NoMoreTransmited:
        ; pending packets still owned by NIC.
        ;--------------------------------------------------------------
        mov     pdescTxRead, bx         ; save the incremented TX
                                        ; descriptor pointer for the next use

TcbNotAvailable:

NothingQueued:
        ret

vPacketTransmitted endp
;****************************************************************************
; Packet Transmitted ended
;****************************************************************************


                align   4

RcvLookahead    proc    near
        mov     Indications,0ffh
        mov     Intpending,NO


        inc     OffDepth

        push    common.cc_ID                    ; Module ID of MAC
        push    cx                              ; Frame Size
        mov     ax, LookAheadSize               ; AX=64
        push    ax                              ; Bytes of lookahead available in buffer
        push    es                              ; Virtual address of lookahead data
        push    di                              ; Virtual address of lookahead data
        push    ds                              ; Virtual address of indicate flag
        lea     ax, DGROUP:indications
        push    ax                              ; Virtual address of indicate flag
        push    protDS                          ; DS of called protocol module
        call    pdispatch.pd_rxlookahead        ; it will call TransferData()

        cli

        mov     ds, cs:cs_ds
        mov     ICPending, YES
        cmp     indications, 0
        je      @F
        dec     OffDepth

@@:
        retn
RcvLookahead    endp

; ------------------------
;


;mmmBREAK <StatusIndicate - Generate Status Indication to Protocol>
;------ StatusIndicate ------------------------------------------------------;
;                                                                            ;
;       void StatusIndicate()                                                ;
;                                                                            ;
;       DX = opcode                                                          ;
;       AX = reason or 0                                                     ;
;                                                                            ;
;       Registers AX,BX,CX,DX,ES and flags may be destroyed.                 ;
;                                                                            ;
;       Generates a status indication to the protocol.                       ;
;                                                                            ;
;----------------------------------------------------------------------------;


StatusIndicate  proc    near

        ; remember that an IndicationComplete needs to be done
        ;--------------------------------------------------------------
        mov     ICpending,YES

        ;-------------------------------
        cmp     indications, 0
        jne     @F

        retn
@@:

        ; cancel any interrupt indication that is pending, since we are allowed to
        ; collapse it (and need to if the protocol is going to leave indications
        ; off) if any other indications are generated
        ;--------------------------------------------------------------
        mov     Intpending,NO

        ; do an implicit IndicationOff.
        ;
        ; increment the IndicationOff depth.  we don't need to actually disable
        ; interrupts from the adapter since we did that earlier.
        ;--------------------------------------------------------------
        inc     offDepth

        ; default indications flag to 0xFF
        ;-------------------------------
        mov     indications,0FFh

        ; issue a status indication to the protocol.  enable interrupts around the
        ; call just in case the protocol spins its wheels a little.
        ;--------------------------------------------------------------
        sti

        push    ax
        push    common.cc_ID
        push    ax
        push    ds
        lea     ax, DGROUP:indications
        push    ax
        push    dx
        push    protDS
        call    pdispatch.pd_status
        pop     ax

        cli

        ; if indications flag wasn't cleared, do an implicit IndicationOn.
        ;
        ; decrement the IndicationOff depth back to zero so that interrupts from
        ; the adapter will be reenabled when we rearm it.
        ;--------------------------------------------------------------
        cmp     indications,0
        jz      @F
        dec     offDepth
@@:
        ret
StatusIndicate  endp


;****************************************************************
; Description :
;   Used for querying and deciding adapter's duplex mode. There
;   will have two ways for calling this function :
;       1. from driver initial
;       2. from link changed
;   -------------------------------------------------------------
; On Entry :
; On Return :
;       LFDXFlag
;
; This procedure is called only at AUTO mode
;--------------------------------------------------------------------

public QueryAuto
QueryAuto      proc    near
        push    bx
        cmp     pciRevisionId, 80h          ; add by Ben (2001/08/06)
        jb      VT3065_MAC2                 ; add by Ben (2001/08/06)

        mov     bx, Ioport

        ; -----------------------------------------
        ; VT3106 machenism (add by Ben, 2001/08/06)
        ; -----------------------------------------
        lea     dx, [bx].SCsrRegStruc.MIISR
        in      ax, dx
        test    ax, W_MIISR_NSPD10          ; 0001h Bit0
        jnz     LSPEED102

        mov     byLineSpeed, 100
        jmp     CheckDuplex2

LSPEED102:
        mov     byLineSpeed, 10

CheckDuplex2:
        test    ax, W_MIISR_NFDX            ; 0004h Bit2
        jnz     FDUPLEX2

        mov     FDXFlag, 0
        mov     FDXFlagNow, 0
        jmp     QueryOK2                    ; check if it's called from INIT or LinkChanged

FDUPLEX2:
        mov     FDXFlag, 1
        mov     FDXFlagNow, 1
        jmp     QueryOK2                    ; check if it's called from INIT or LinkChanged

        ; -----------------------------------------
        ;       VT3065 or VT3043 machenism
        ; -----------------------------------------
VT3065_MAC2:                                ; add by Ben (2001/08/06)
        ; added by Ben (10/04/00), Line speed should be detected after re-auto!
        ; ---------------------------------------------------------------------\
        ; Check MIISR(6dh) port to know linespeed after re-auto
        ; ---------------------------------------------------------------------
        lea     dx, [bx].SCsrRegStruc.MIISR
        in      al, dx
        test    al, 01h
        jz      MII_100M2

        mov     byLineSpeed, 10
        jmp     Set_Line_Speed_OK2
MII_100M2:
        mov     byLineSpeed, 100
Set_Line_Speed_OK2:
        ; ---------------------------------------------------------------------/

        ;VT3065A : No SEEQ <06>
        ;-------------------------------
        cmp     pciRevisionId, 40h
        jne     ExitSeeqPHY2
        mov     byMIIIndex, 06h ; Auto Negotiation Expansion Register, ANER
        call    MIIRead
        test    ax, 0001h       ; Link partner has N-WAY ability?
        jz      NoNWay2
ExitSeeqPHY2:
        ;VT3065A : No SEEQ <06>
        ;-------------------------------

        mov     byMIIIndex, MII_REG_ANLPAR ; Link partner ability Register, ANLPAR
        call    MIIRead
        test    ax, 0140h       ; 100_FD OR 10_FD ?
        jz      NoNWay2
        mov     wBak, ax

        mov     byMIIIndex, MII_REG_ANAR ; Do I have the Full-Duplex ability, ANAR
        call    MIIRead
        test    ax, 0140h       ; 100_FD OR 10_FD ?
        jz      NoNWay2

        ; added by Ben (10/04/00)
        ; We can't use BMCR Bit13 to check the selected media type, although
        ; Davicom and SEEQ PHY can reflect the SPEED, but NS's PHY will ignore
        ; this Bit after auto-negotiation completed. ( by Ben, 10/04/00)
        ; --------------------------------------------------------------------
        cmp     byLineSpeed, 100
        jne     Not1002

        and     ax, 0100h       ; masked the local of 10-F ability
        jmp     Label12
Not1002:

        and     ax, 0040h       ; masked the local of 100-F ability
Label12:

        test    ax, wBak        ; 0100h and 0040h ? else OK (wBak->ANLPAR)
        jz      NoNWay2         ; 10-F vs 100-F
        mov     FDXFlag, 1
        mov     FDXFlagNow, 1
        jmp     QueryOK2

NoNWay2:
        cmp     AutoFlag, 1
        jz      QueryOK11

        mov     byMIIIndex, MII_REG_BMCR
        mov     wMIIValue, BMCR_AUTO
        call    MIITurnOffBits
        call    MIIDelay

        cmp     byUserOverride, 1       ; user specify SPEED or Duplex keywork
        jnz     QueryOK11

        mov     al, FDXFlag
        mov     FDXFlagNow, al
        jmp     QueryOK2

QueryOK11:
        mov     FDXFlag, 0
        mov     FDXFlagNow, 0

QueryOK2:
        pop     bx
        ret
QueryAuto      endp

;------------------------------------------------------------
; Set Half Duplex mode
;
;------------------------------------------------------------
public  SetToHalfDPX
SetToHalfDPX   proc    near
        push    bx
        mov     bx, Ioport
        lea     dx, [bx].SCsrRegStruc.CommandReg
        in      ax, dx
        and     ax, not 0400h   ; full DUPLEX off
        out     dx, ax

        pop     bx
        ret

SetToHalfDPX   endp

; ------------------------------------------------------------
;
;
; ------------------------------------------------------------
public SetToFullDPX
SetToFullDPX   proc    near
        push    bx
        mov     bx, Ioport
        lea     dx, [bx].SCsrRegStruc.CommandReg

        in      ax, dx
        or      ax, 0400h       ; full DUPLEX on
        out     dx, ax

        pop     bx
        ret
SetToFullDPX   endp

MIIDelay       proc
        push    ecx
        push    eax
        mov     ecx, 01fffh
 WaitingRead2:
        in      al, 61h
        in      al, 61h
        in      al, 61h
        loop    WaitingRead2
        pop     eax
        pop     ecx
        ret
MIIDelay       endp

IFDEF   LinkChange

SenseLinkUnLink proc    near

        push    bx

        mov     bx, Ioport

        lea     dx, [bx].SCsrRegStruc.MIISR

        in      al, dx
        test    al, 02h                   ; Link Fail Bit(LNKFL)
        jz      LinkNow2
        mov     LinkStatus, UnLink
        jmp     SetLinkFlagOver2
 LinkNow2:
        mov     LinkStatus, LinkOK
 SetLinkFlagOver2:
        pop     bx
        ret
SenseLinkUnLink endp
ENDIF

; ----------------------------------------------------------
; EnableMiiAutoPolling
; Input:
;   Ioport : I/O base address
; OUTPUT:
;   none
;   all register will be reserved
; ----------------------------------------------------------------
EnableMiiAutoPolling  proc    near
    push    ax
    push    bx
    push    cx
    push    dx

    mov     bx, Ioport
    ; modified by AJ.
    ; turn off MAUTO
    lea     dx, [bx].SCsrRegStruc.MIICR             ;70h
    mov     al, 0
    out     dx, al

    ; write MAD0~MAD4 to 1 in MIIADR
    lea     dx, [bx].SCsrRegStruc.MIIADR            ;71h
    mov     al, 01
    out     dx, al

    ; turn on MAUTO
    lea     dx, [bx].SCsrRegStruc.MIICR             ;70h
    mov     al, BY_MIICR_MATUO
    out     dx, al

    ; as soon as MDONE is on, MAUTO is really started
    mov     cx, MAX_TIMEOUT
    lea     dx, [bx].SCsrRegStruc.MIIADR            ;71h
Waiting1:
    in      al, dx
    test    al, BY_MIIADR_MDONE
    jnz     MDone
    loop    Waiting1
MDone:

    ; turn on MSRCEN
    lea     dx, [bx].SCsrRegStruc.MIIADR            ;71h
    in      al, dx
    or      al, BY_MIIADR_MSRCEN                    ; turn on MSRCEN
    out     dx, al

    pop     dx
    pop     cx
    pop     bx
    pop     ax
    ret
EnableMiiAutoPolling  endp


;---------------------------------------------
; SafeDisableMiiAutoPoll
;   turn of MAUTO
;   IN:
;       Ioport: I/O address
;       pciRevisionId: Rev ID
;   OUTPUT:
;       none
;   all register will be reserved
;----------------------------------------------
SafeDisableMiiAutoPoll  proc    near
    push    ax
    push    bx
    push    cx
    push    dx

    mov     bx, Ioport

    ; turn off MAUTO
    lea     dx, [bx].SCsrRegStruc.MIICR     ; 70h
    mov     al, 0
    out     dx, al

    ; waiting MIDEL on, as soon as MIDEL is on, MAUTO is really stoped
    mov     cx, MAX_TIMEOUT
    lea     dx, [bx].SCsrRegStruc.MIIADR
WaitingMIDELOn:
    in      al, dx
    test    al, BY_MIIADR_MIDLE
    jnz     MIDELisOn
    loop    WaitingMIDELOn

MIDELisOn:
    jmp     SafeDisableMiiAutoPollExit

SafeDisableMiiAutoPollVT3043:

SafeDisableMiiAutoPollExit:
    pop     dx
    pop     cx
    pop     bx
    pop     ax
    ret

SafeDisableMiiAutoPoll  endp


;--------------------------------------------------
; MIITurnOnBits
;   Turn on bits of MII register
;   IN:
;       Ioport: I/O base address
;       byMIIIndex: the MII register that be wrote
;       wMIIValue: mask
;   OUTPUT:
;       none
;   all register will be reserved
;--------------------------------------------------------------

public MIITurnOnBits
MIITurnOnBits   proc    near

    push    ax

    call    MIIRead

    or      ax, wMIIValue
    mov     wMIIValue, ax

    call    MIIWrite

    pop     ax
    ret

MIITurnOnBits   endp

;--------------------------------------------------
; MIITurnOffBits
;   Turn off bits of MII register
;   IN:
;       Ioport: I/O base address
;       byMIIIndex: the MII register that be wrote
;       wMIIValue: mask
;   OUTPUT:
;       none
;   all register will be reserved
;--------------------------------------------------------------

public MIITurnOffBits
MIITurnOffBits  proc    near
    push    ax
    push    cx

    call    MIIRead

    mov     cx, wMIIValue
    not     cx
    and     ax, cx
    mov     wMIIValue, ax

    call    MIIWrite

    pop     cx
    pop     ax
    ret
MIITurnOffBits  endp

;--------------------------------------------------
; MIIWrite
;   write data to MII register
;   IN:
;       Ioport: I/O base address
;       byMIIIndex: the MII register that be wrote
;       wMIIValue: the data that wrote
;   OUTPUT:
;       if fail, zflag is set
;   all register will be reserved
;--------------------------------------------------------------
MIIWrite    proc    near
    push    ax
    push    bx
    push    cx
    push    dx

    call    SafeDisableMiiAutoPoll

    mov     bx, Ioport
    lea     dx, [bx].SCsrRegStruc.MIIADR            ;71h
    mov     al, byMIIIndex
    out     dx, al

    mov     ax, wMIIValue
    lea     dx, [bx].SCsrRegStruc.MII_DATA_REG
    out     dx, ax

    ; turn on write command
    lea     dx, [bx].SCsrRegStruc.MIICR             ;70h
    in      al, dx
    or      al, BY_MIICR_WCMD
    out     dx, al

    ; wait WCMD off, as soon as WCMD is off, write command is finished
    mov     cx, MAX_TIMEOUT
WaitWCMDOff:
    in      al, dx
    test    al, BY_MIICR_WCMD
    jz      WCMDOff
    loop    WaitWCMDOff
WCMDOff:

    call    EnableMiiAutoPolling


    ; if fail, cx is 0. set zero flag
    or      cx, cx

    pop     dx
    pop     cx
    pop     bx
    pop     ax
    ret
MIIWrite    endp
;--------------------------------------------------------
; MIIRead
;   read MII register
;   IN:
;       Ioport: I/O Base address
;       byMIIIndex: address of register of MII
;   OUTPUT:
;       if fail, zflag is be set, else zflag is clear
;       ax: data read
;   NOTE:
;       all register will be reserved except ax
;------------------------------------------------------------

MIIRead     proc    near
    push    bx
    push    cx
    push    dx

    ; disable MIICR_MAUTO, so that mii addr can be set normally
    call    SafeDisableMiiAutoPoll

    mov     bx, Ioport
    ; set MII register offset
    lea     dx, [bx].SCsrRegStruc.MIIADR
    in      al, dx
    ;----------------------------------------------------------------
    ; 07/25/2003 added by guard
    ; fixed the bug.
    ; original code
    ;   mov ah, byMIIIndex
    ;   or  al, ah
    ; this issue if MAUTO is turned off and MAD0~MAD4 isn't clear
    ; we should read wrong register
    ; fixed code
    ;   mov ah, byMIIIndex
    ;   and al, 0E0h
    ;   or  al, ah
    ;------------------------------------------------------------------
    mov     ah, byMIIIndex
    and     al, 0E0h
    or      al, ah

    out     dx, al

    ; turn on Read command
    lea     dx, [bx].SCsrRegStruc.MIICR             ;70h
    in      al, dx
    or      al, BY_MIICR_RCMD
    out     dx, al

    ; waiting RCMD off, as soon as RCMD is off, the read data is in MIIData
    mov     cx, MAX_TIMEOUT
WaitRCMDisOff:
    in      al, dx
    test    al, BY_MIICR_RCMD
    jz      RCMDisOff
    loop    WaitRCMDisOff
RCMDisOff:

    lea     dx, [bx].SCsrRegStruc.MII_DATA_REG
    in      ax, dx

    call    EnableMiiAutoPolling

    ; if fail, cx is 0. set zero flag
    or      cx, cx

    pop     dx
    pop     cx
    pop     bx
    ret
MIIRead     endp
                END_CODE

                end
